<?php
/**
 * HelloPack Modules Admin Page
 *
 * @package HelloPack_Client
 * @since 2.1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Modules admin class.
 */
class HelloPack_Client_Modules_Admin {

	/**
	 * Constructor
	 */
	public function __construct() {
		// Add AJAX handlers
		add_action( 'wp_ajax_hellopack_toggle_module', array( $this, 'ajax_toggle_module' ) );
		
		// Load scripts on the HelloPack admin page
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
	}

	/**
	 * Enqueue admin scripts
	 *
	 * @param string $hook Current admin page.
	 */
	public function enqueue_scripts( $hook ) {
		// Load on main HelloPack page
		if ( 'toplevel_page_hellopack-client' !== $hook && 'hellopack-client_page_hellopack-client' !== $hook ) {
			return;
		}

		// Only load if we're on the modules tab
		if ( ! isset( $_GET['tab'] ) || 'modules' !== $_GET['tab'] ) {
			return;
		}

		wp_enqueue_script( 'hellopack-client-modules', HELLOPACK_CLIENT_URI . 'js/modules-admin.js', array( 'jquery' ), HELLOPACK_CLIENT_VERSION, true );
	}

	/**
	 * AJAX handler for toggling module
	 */
	public function ajax_toggle_module() {
		// Check nonce
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'hellopack_modules_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed', 'hellopack-client' ) ) );
		}

		// Check capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized', 'hellopack-client' ) ) );
		}

		// Get module ID and enabled status
		$module_id = isset( $_POST['module_id'] ) ? sanitize_text_field( $_POST['module_id'] ) : '';
		$enabled = isset( $_POST['enabled'] ) && $_POST['enabled'] === 'true' ? true : false;

		if ( empty( $module_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid module ID', 'hellopack-client' ) ) );
		}

		// Get current settings
		$settings = get_option( 'hellopack_modules_settings', array() );
		
		// Update module setting
		$settings[ $module_id ] = array(
			'enabled' => $enabled,
			'updated' => current_time( 'mysql' ),
		);

		// Save settings
		update_option( 'hellopack_modules_settings', $settings );

		// Get module instance to check if it's active
		$module_loader = HelloPack_Module_Loader::get_instance();
		$module = $module_loader->get_module( $module_id );
		
		$is_active = false;
		if ( $module && $enabled ) {
			$is_active = $module->should_load();
		}

		// Send success response
		wp_send_json_success( array(
			'message' => $enabled ? __( 'Module enabled successfully.', 'hellopack-client' ) : __( 'Module disabled successfully.', 'hellopack-client' ),
			'is_active' => $is_active,
		) );
	}

	/**
	 * Render modules page
	 */
	public function render_modules_page() {
		$module_loader = HelloPack_Module_Loader::get_instance();
		$modules = $module_loader->get_modules();
		$settings = $module_loader->get_module_settings();
		
		// Group modules by category
		$grouped_modules = array();
		foreach ( $modules as $module ) {
			$category = $module->get_category();
			if ( ! isset( $grouped_modules[ $category ] ) ) {
				$grouped_modules[ $category ] = array();
			}
			$grouped_modules[ $category ][] = $module;
		}
		
		// Sort categories
		ksort( $grouped_modules );
		?>
		<div class="wrap">
			<h1><?php _e( 'HelloPack Modules', 'hellopack-client' ); ?></h1>
			
			<?php if ( isset( $_GET['updated'] ) && $_GET['updated'] === 'true' ) : ?>
				<div class="notice notice-success is-dismissible">
					<p><?php _e( 'Module settings updated.', 'hellopack-client' ); ?></p>
				</div>
			<?php endif; ?>

			<p class="description">
				<?php _e( 'Modules are only displayed when their target plugin or theme is active. You can manually disable modules here if needed.', 'hellopack-client' ); ?>
			</p>
			<p class="description">
				<strong><?php _e( 'Status indicators:', 'hellopack-client' ); ?></strong>
				<span class="status-active" style="margin-left: 10px;">●</span> <?php _e( 'Active - Module is enabled and running', 'hellopack-client' ); ?>
				<span class="status-ready" style="margin-left: 10px;">●</span> <?php _e( 'Ready - Module is enabled but not running', 'hellopack-client' ); ?>
				<span class="status-disabled" style="margin-left: 10px;">●</span> <?php _e( 'Disabled - Module is manually disabled', 'hellopack-client' ); ?>
			</p>

			<div class="hellopack-modules-container">
				<?php foreach ( $grouped_modules as $category => $category_modules ) : ?>
					<div class="hellopack-module-category">
						<h2 class="category-title">
							<?php echo esc_html( ucfirst( $category ) . ' ' . __( 'Modules', 'hellopack-client' ) ); ?>
							<span class="count">(<?php echo count( $category_modules ); ?>)</span>
						</h2>
						
						<table class="wp-list-table widefat fixed striped">
							<thead>
								<tr>
									<th scope="col" class="manage-column column-cb check-column">
										<span class="screen-reader-text"><?php _e( 'Enable/Disable', 'hellopack-client' ); ?></span>
									</th>
									<th scope="col" class="manage-column column-name"><?php _e( 'Module', 'hellopack-client' ); ?></th>
									<th scope="col" class="manage-column column-target"><?php _e( 'Target', 'hellopack-client' ); ?></th>
									<th scope="col" class="manage-column column-status"><?php _e( 'Status', 'hellopack-client' ); ?></th>
									<th scope="col" class="manage-column column-actions"><?php _e( 'Actions', 'hellopack-client' ); ?></th>
								</tr>
							</thead>
							<tbody>
								<?php foreach ( $category_modules as $module ) : 
									$module_info = $module->get_info();
									$module_id = $module_info['id'];
									$is_enabled = ! isset( $settings[ $module_id ] ) || $settings[ $module_id ]['enabled'] !== false;
									$is_active = $module_info['active'];
									$target = $module_info['target'];
									?>
									<tr>
										<th scope="row" class="check-column">
											<input type="checkbox" 
												   class="module-toggle" 
												   data-module-id="<?php echo esc_attr( $module_id ); ?>"
												   <?php checked( $is_enabled ); ?> />
										</th>
										<td class="column-name">
											<strong><?php echo esc_html( $module_info['name'] ); ?></strong>
											<br />
											<span class="description"><?php echo esc_html( $module_info['description'] ); ?></span>
										</td>
										<td class="column-target">
											<?php 
											if ( 'plugin' === $category && isset( $target['plugin_dir'] ) ) {
												echo esc_html( $target['plugin_dir'] );
											} elseif ( 'theme' === $category && isset( $target['theme_slug'] ) ) {
												echo esc_html( $target['theme_slug'] );
											} else {
												echo '—';
											}
											?>
										</td>
										<td class="column-status">
											<?php if ( $is_active && $is_enabled ) : ?>
												<span class="status-active"><?php _e( 'Active', 'hellopack-client' ); ?></span>
											<?php elseif ( ! $is_enabled ) : ?>
												<span class="status-disabled"><?php _e( 'Disabled', 'hellopack-client' ); ?></span>
											<?php else : ?>
												<span class="status-ready"><?php _e( 'Ready', 'hellopack-client' ); ?></span>
											<?php endif; ?>
										</td>
										<td class="column-actions">
											<form method="post" style="display: inline;">
												<?php wp_nonce_field( 'hellopack_module_toggle' ); ?>
												<input type="hidden" name="hellopack_module_toggle" value="1" />
												<input type="hidden" name="module_id" value="<?php echo esc_attr( $module_id ); ?>" />
												<?php if ( $is_enabled ) : ?>
													<button type="submit" class="button button-small"><?php _e( 'Disable', 'hellopack-client' ); ?></button>
												<?php else : ?>
													<input type="hidden" name="enabled" value="1" />
													<button type="submit" class="button button-small"><?php _e( 'Enable', 'hellopack-client' ); ?></button>
												<?php endif; ?>
											</form>
										</td>
									</tr>
								<?php endforeach; ?>
							</tbody>
						</table>
					</div>
				<?php endforeach; ?>
				
				<?php if ( empty( $modules ) ) : ?>
					<div class="notice notice-info">
						<p><?php _e( 'No modules found. Modules will appear here once they are added to the modules directory.', 'hellopack-client' ); ?></p>
					</div>
				<?php endif; ?>
			</div>
		</div>

		<style>
			.hellopack-modules-container {
				margin-top: 20px;
			}
			.hellopack-module-category {
				margin-bottom: 40px;
			}
			.category-title {
				font-size: 1.3em;
				margin-bottom: 10px;
			}
			.category-title .count {
				color: #666;
				font-weight: normal;
			}
			.column-cb {
				width: 50px;
			}
			.column-target {
				width: 20%;
			}
			.column-status {
				width: 15%;
			}
			.column-actions {
				width: 15%;
			}
			.status-active {
				color: #46b450;
				font-weight: bold;
			}
			.status-disabled {
				color: #dc3232;
			}
			.status-ready {
				color: #0073aa;
			}
		</style>
		<?php
	}
}