<?php
/**
 * WP Rocket Module
 *
 * @package HelloPack_Client
 * @since 2.1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WP Rocket module class.
 */
class HelloPack_Module_WP_Rocket extends HelloPack_Module_Base {

	/**
	 * Setup module properties
	 */
	protected function setup() {
		$this->id          = 'wp-rocket';
		$this->name        = __( 'WP Rocket', 'hellopack-client' );
		$this->description = __( 'Manages WP Rocket updates through centralized updater.', 'hellopack-client' );
		$this->category    = 'plugin';
		$this->version     = '1.0.0';
		$this->target      = array(
			'plugin_dir'  => 'wp-rocket',
			'plugin_file' => 'wp-rocket.php',
		);
	}

	/**
	 * Initialize the module
	 */
	public function init() {
		// Setup WP Rocket immediately since plugins_loaded has already fired
		$this->setup_wp_rocket();

		// Set customer data transient
		$this->set_customer_data();

		// Override update check API
		$this->override_update_api();
	}

	/**
	 * Setup WP Rocket configuration
	 */
	public function setup_wp_rocket() {
		// Set API servers for valid key check
		if ( class_exists( 'HPack_Set_API_Servers' ) ) {
			$wprocket_get = new HPack_Set_API_Servers();
			$wprocket_get->set_api_servers( 'api.wp-rocket.me/valid_key.php', 'api-register.wp-json.app/wprocket/set/?rout=' );
			$wprocket_get->init();
		}

		// Set API servers for license activation
		if ( class_exists( 'HPack_Set_API_Servers' ) ) {
			$wprocket_set = new HPack_Set_API_Servers();
			$wprocket_set->set_api_servers( 'api.wp-rocket.me/api/wp-rocket/activate-licence.php', 'api-register.wp-json.app/wprocket/set/?rout=' );
			$wprocket_set->init();
		}

		// Set API servers for user stats
		if ( class_exists( 'HPack_Set_API_Servers' ) ) {
			$wprocket_user = new HPack_Set_API_Servers();
			$wprocket_user->set_api_servers( 'api.wp-rocket.me/stat/1.0/wp-rocket/user.php', 'api-register.wp-json.app/wprocket/user/?rout=' );
			$wprocket_user->init();
		}

		// Define WP Rocket constants if not already defined
		if ( ! defined( 'WP_ROCKET_KEY' ) ) {
			define( 'WP_ROCKET_KEY', 'x3llx146' );
		}

		if ( ! defined( 'WP_ROCKET_EMAIL' ) ) {
			define( 'WP_ROCKET_EMAIL', 'support@hellowp.io' );
		}
	}

	/**
	 * Set WP Rocket customer data transient
	 */
	private function set_customer_data() {
		// Get WP Rocket version if available
		$wp_rocket_version = defined( 'WP_ROCKET_VERSION' ) ? WP_ROCKET_VERSION : '3.17.3';

		// Get site name or use default
		$site_name = get_bloginfo( 'name' );
		$first_name = ! empty( $site_name ) ? $site_name : 'HelloPack';

		// Get admin email
		$admin_email = get_option( 'admin_email' );
		if ( empty( $admin_email ) ) {
			$admin_email = 'support@hellowp.io';
		}

		// License key from HelloPack
		$license_key = defined( 'HP_GLOBAL_SERIAL' ) ? HP_GLOBAL_SERIAL : 'x3llx146';

		// Create customer data object
		$customer_data = new stdClass();
		$customer_data->licence_version = $wp_rocket_version;
		$customer_data->consumer_key = $license_key;
		$customer_data->first_name = $first_name;
		$customer_data->email = $admin_email;
		$customer_data->licence_account = '-1';
		$customer_data->licence_expiration = strtotime( '+10 years' ); // 10 years from now
		$customer_data->ID = '1333';
		$customer_data->is_blacklisted = '';
		$customer_data->is_staggered = '';
		$customer_data->date_created = '0';
		$customer_data->status = 'active';
		$customer_data->is_blocked = '';
		$customer_data->has_auto_renew = '';
		$customer_data->{'has_one-com_account'} = '';
		$customer_data->renewal_url = '';
		$customer_data->upgrade_plus_url = '';
		$customer_data->upgrade_infinite_url = '';

		// Set transient for 1 year (WP Rocket caches this for a long time)
		set_transient( 'wp_rocket_customer_data', $customer_data, YEAR_IN_SECONDS );

		// Hide analytics notice
		if ( function_exists( 'HP_check_options' ) ) {
			HP_check_options( 'rocket_analytics_notice_displayed', 1 );
		}
	}

	/**
	 * Override WP Rocket update check API
	 */
	private function override_update_api() {
		if ( class_exists( 'HPack_Set_API_Servers' ) ) {
			$wprocket = new HPack_Set_API_Servers();
			$wprocket->over_api_servers( 'api.wp-rocket.me/check_update.php' );
			$wprocket->init();
		}
	}
}